#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <sys/inotify.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>
#include <limits.h>
#include <errno.h>
#include <regex.h>
#include <sys/select.h>

#define MAX_DIRS 128

typedef struct _dirInfo {
	char *name;
	int fd;
	int wd;
} dirInfo;

int printHelp = 0;
int isRecursive = 0;
char *dirname = NULL;
char *pattern;
uid_t uid;
gid_t gid;
regex_t compiledRE;
dirInfo *dirsToWatch[MAX_DIRS];
int numDirs = 0;
uint32_t watchFlags = IN_MOVE_SELF|IN_DELETE_SELF|IN_DONT_FOLLOW;
int oneEvent = 0;
int timeout = 0;

/* Prototypes */
void usage(char *argv0, const char *msg);
const char *checkargs(int argc, char *argv[]);
int collectDirs(char *name);
int initWatchDirs(fd_set *rfds);
int watchDirs(void);
int closeWatchDirs(void);


void usage(char *argv0, const char *msg)
{
	if (msg != NULL) {
		printf("ERROR: %s\n", msg);
	}
	printf("Usage: %s [-h|--help] [-R|--recursive] dirname [[-p|--pattern] pattern] [[-w|--watch] flags] [-t|--timeout s]\n", argv0);
	printf("\n");
	printf("The dirname denotes the directory to be watched for file changes");
	printf("The optional pattern specifies a pattern file names must match\n");
	printf("The -R (--recursive) flag operates recursively (use with care)\n");
	printf("The -w flag specifies the events to watch for\n");
	printf("     flags that can be used are:\n");
	printf("     A = File Attributes Change\n");
	printf("     C = File Creation\n");
	printf("     D = File Deletion\n");
	printf("     M = File Modification\n");
	printf("     R = File Rename (Move)\n");
	printf("     1 = Stop after first event\n");
	printf("     The flags are case insensitive\n");
	printf("     If no flags are specified, watching for file creation is the default behaviour\n");
	printf("The -t flag is used to specify a timeout in seconds\n");
	printf("     If no timeout is specified, the process will wait indefinitely (unless it stops after the first event)\n");
	printf("The -h (--help) option displays this message\n");
	printf("Symbolic links aren't followed\n");
	return;
}

const char *checkargs(int argc, char *argv[])
{
	int gotDir = 0;
	int gotR = 0;
	int gotW = 0;
	int gotP = 0;
	int gotT = 0;
	int i;
	struct stat statbuf;
	int accessOK;
	int status;
	char *flags;
	char *errchar;

	if (argc < 2)
		return "Missing directory name\n";
	for (i = 1; i < argc; ++i) {
		if (!strncmp(argv[i], "-h", 2) || !strncmp(argv[i], "--help", 6)) {
			printHelp = 1;
			break;
		}
		if (!strncmp(argv[i], "-R", 2) || !strncmp(argv[i], "--recursive", 12)) {
			if (gotR) {
				return "Duplicate --recursive option\n";
			}
			isRecursive = 1;
			gotR = 1;
			continue;
		}
		if (!strncmp(argv[i], "-p", 2) || !strncmp(argv[i], "--pattern", 9)) {
			if (gotP) {
				return "Duplicate --pattern option\n";
			}
			i++;
			if (i == argc)
				return "Pattern missing\n";
			pattern = strdup(argv[i]);
			if (pattern == NULL) {
				return "Memory allocation error\n";
			}
			status = regcomp (&compiledRE, pattern, REG_EXTENDED);
			if (status != 0) {
				return "Error compiling the provided pattern (%s)\n";
			}
			gotP = 1;
			continue;
		}
		if (!strncmp(argv[i], "-w", 2) || !strncmp(argv[i], "--watch", 7)) {
			if (gotW) {
				return "Duplicate --watch option\n";
			}
			i++;
			if (i == argc)
				return "Watch flags missing\n";
			flags = argv[i];
			
			while(*flags != '\0') {
				switch (*flags) {
					case 'a':
					case 'A':
						watchFlags |= IN_ATTRIB;
						break;
					case 'c':
					case 'C':
						watchFlags |= IN_CREATE|IN_CLOSE_WRITE;
						break;
					case 'd':
					case 'D':
						watchFlags |= IN_DELETE;
						break;
					case 'm':
					case 'M':
						watchFlags |= IN_MODIFY|IN_CLOSE_WRITE;
						break;
					case 'r':
					case 'R':
						watchFlags |= IN_MOVED_TO|IN_MOVED_FROM;
						break;
					case '1':
						oneEvent = 1;
						break;
					default:
						return "Unknown flag\n";
				}
				flags++;
			}
			gotW = 1;
		}
		if (!strncmp(argv[i], "-t", 2) || !strncmp(argv[i], "--timeout", 9)) {
			if (gotT) {
				return "Duplicate --timeout option\n";
			}
			i++;
			if (i == argc)
				return "Timeout specification missing\n";

			timeout = (int) strtol(argv[i], &errchar, 10);
			if (*errchar != '\0')
				return "Invalid number format\n";
			gotT = 1;
		}
		if (!gotDir) {
			if (gotDir) {
				return "More than one directory specified\n";
			}
			dirname = strdup(argv[i]);
			if (dirname == NULL) {
				return "Memory allocation error\n";
			}
			gotDir = 1;
			continue;
		}
	}

	if (!gotDir) {
		return "No directory specified\n";
	}
	if (!gotW) {
		watchFlags |= IN_CREATE|IN_CLOSE_WRITE;
	}

	if (lstat(dirname, &statbuf) < 0) {
		fprintf(stderr, "stat error : %s\n", strerror(errno));
		return "Error in call to stat()\n";
	}
	if (S_ISDIR(statbuf.st_mode) == 0) {
		fprintf(stderr, "The specified path (%s) isn't a directory", dirname);
		return "The specified path isn't a directory\n";
	}

	accessOK = 0;
	if (uid == statbuf.st_uid) {	/* we're owner */
		if ((statbuf.st_mode & (S_IRUSR | S_IXUSR)) == (S_IRUSR | S_IXUSR))
			accessOK = 1;	/* read / chdir access to directory */
	} 
	if (gid == statbuf.st_gid) {	/* we're in the same group */
		if ((statbuf.st_mode & (S_IRGRP | S_IXGRP)) == (S_IRGRP | S_IXGRP))
			accessOK = 1;	/* read / chdir access to directory */
	}
					/* we're world/others */
	if ((statbuf.st_mode & (S_IROTH | S_IXOTH)) == (S_IROTH | S_IXOTH))
		accessOK = 1;

	if (!accessOK)
		return "Directory not accessible\n";

	return NULL;
}

int collectDirs(char *name)
{
	char *actdir;
	struct stat statbuf;
	struct dirent *dirp;
	DIR  *dp;
	int accessOK;
	int rc = 0;

	/* add to list of dirs to watch */
	if (numDirs >= MAX_DIRS) {
		fprintf(stderr, "Maximum number of directories (%d) exceeded\n", MAX_DIRS);
		return -1;
	}
	dirsToWatch[numDirs] = malloc(sizeof(dirInfo));
	if (dirsToWatch[numDirs] == NULL) {
		fprintf(stderr, "Memory allocation error\n");
	}
	dirsToWatch[numDirs]->name = name;
	dirsToWatch[numDirs]->fd = -1;
	dirsToWatch[numDirs]->wd = -1;
	numDirs++;

	/* if recursive ... */
	if (isRecursive) {
		actdir = malloc((PATH_MAX + 1) * sizeof(char));
		if (actdir == NULL) {
			fprintf(stderr, "Memory allocation error\n");
			return -1;
		}
		if ((dp = opendir(name)) != NULL) {		/* we igore the directory if we cannot open it */
			while ((dirp = readdir(dp)) != NULL) {
				if (!strcmp(dirp->d_name, ".") || !strcmp(dirp->d_name, ".."))
					continue;		/* ignore "." and ".." directories */

				if (snprintf(actdir, PATH_MAX + 1, "%s/%s", name, dirp->d_name) > PATH_MAX) {
					fprintf(stderr, "Path buffer overflow");
					return -1;
				}
				if (lstat(actdir, &statbuf) < 0)
					continue;		/* can't stat(), ignore */
				if (S_ISDIR(statbuf.st_mode) == 0)
					continue;		/* ignore non directories */

				accessOK = 0;
				if (uid == statbuf.st_uid) {	/* we're owner */
					if ((statbuf.st_mode & (S_IRUSR | S_IXUSR)) == (S_IRUSR | S_IXUSR))
						accessOK = 1;	/* read / chdir access to directory */
				} 
				if (gid == statbuf.st_gid) {	/* we're in the same group */
					if ((statbuf.st_mode & (S_IRGRP | S_IXGRP)) == (S_IRGRP | S_IXGRP))
						accessOK = 1;	/* read / chdir access to directory */
				}
				 				/* we're world/others */
				if ((statbuf.st_mode & (S_IROTH | S_IXOTH)) == (S_IROTH | S_IXOTH))
					accessOK = 1;
				
				if (accessOK)			/* descend recursively */
					if (collectDirs(actdir) != 0) {
						rc = -1;
						break;		/* some error */
					}
			}

			closedir(dp);
		}
	}

	return rc;
}

int initWatchDirs(fd_set *rfds)
{
	int i;
	int maxfd = 0;

	FD_ZERO (rfds);

	dirsToWatch[i]->fd = inotify_init1(IN_NONBLOCK);
	if (dirsToWatch[i]->fd < 0) {
		fprintf(stderr, "Error opening watch channel for directory %s\n", dirsToWatch[i]->name);
		return -1;
	}
	FD_SET (dirsToWatch[i]->fd, rfds);
	if (dirsToWatch[i]->fd > maxfd) maxfd = dirsToWatch[i]->fd;

	for (i = 0; i < numDirs; ++i) {
		dirsToWatch[i]->wd = inotify_add_watch (dirsToWatch[i]->fd, dirsToWatch[i]->name, watchFlags);
		if (dirsToWatch[i]->wd < 0) {
			fprintf(stderr, "Error opening watch channel for directory %s\n", dirsToWatch[i]->name);
			return -1;
		}
	}

	return maxfd + 1;
}

int closeWatchDirs(void)
{
	int rc = 0;
	int i;

	for (i = 0; i < numDirs; ++i) {
		if (dirsToWatch[i]->wd >= 0)
			inotify_rm_watch(dirsToWatch[i]->fd, dirsToWatch[i]->wd);
		if (dirsToWatch[i]->fd >= 0)
			close(dirsToWatch[i]->fd);
	}

	return rc;
}

int watchDirs(void)
{
	int maxfd;
	int rc;
	int fd;
	int i;
	fd_set rfds;
	fd_set workrfds;
	struct timeval time;
	struct timeval *timeToUse = NULL;
	char *buffer;
	int bufferSize = 1024 * (sizeof(struct inotify_event) + 32 /* estimated average name length */);
	int numRead;
	struct inotify_event *eventptr;
	int pos;

	buffer = malloc(bufferSize);
	if (buffer == NULL) {
		fprintf(stderr, "Memory allocation error\n");
		return -1;
	}

	maxfd = initWatchDirs(&rfds);
	if (maxfd < 0)
		return -1;

	if (timeout != 0) {
		time.tv_usec = 0;
		time.tv_sec = timeout;
		timeToUse = &time;
	}
	while (1) {
		memcpy(&workrfds, &rfds, sizeof(fd_set));	/* restore the original set of fd */
		rc = select(maxfd, &workrfds, NULL, NULL, timeToUse);
		if (rc < 0) {
			fprintf(stderr, "Error occurred during select() system call : %s\n", strerror(errno));
			break;
		}
		if (rc == 0) {
			/* run into timeout */
			break;
		}
		for (fd = 0; fd < maxfd; ++fd) {
			if (FD_ISSET (fd, &workrfds)) {
				/* search directory to watch; a linear search is good enough */
				for (i = 0; i < numDirs; ++i) {
					if (fd == dirsToWatch[i]->fd)
						break;
				}
				/* we only read once; if more events are in the queue, the select will wake up immediately */
				numRead = read(fd, buffer, bufferSize);
				if (numRead < 0) {
					fprintf(stderr, "Error occurred while reading the event queue: %s\n", strerror(errno));
					break;
				}
				pos = 0;
				while (pos < numRead) {
					eventptr = (struct inotify_event *) (buffer + pos);
					/*
						IN_ACCESS         File was accessed (read) (*).
						IN_ATTRIB         Metadata changed, e.g., permissions, timestamps, extended attributes, link count (since Linux 2.6.25), UID, GID, etc. (*).
						IN_CLOSE_WRITE    File opened for writing was closed (*).
						IN_CLOSE_NOWRITE  File not opened for writing was closed (*).
						IN_CREATE         File/directory created in watched directory (*).
						IN_DELETE         File/directory deleted from watched directory (*).
						IN_DELETE_SELF    Watched file/directory was itself deleted.
						IN_MODIFY         File was modified (*).
						IN_MOVE_SELF      Watched file/directory was itself moved.
						IN_MOVED_FROM     Generated for the directory containing the old filename when a file is renamed (*).
						IN_MOVED_TO       Generated for the directory containing the new filename when a file is renamed (*).
						IN_OPEN           File was opened (*).
					*/
					printf("Directory: %s\n", dirsToWatch[i]->name);
					printf("File event registered:\nname=%s\n", eventptr->len == 0 ? "<not specified>" : eventptr->name);
					if ((eventptr->mask & IN_ACCESS) != 0)
						printf("\tFile was accessed\n");
					if ((eventptr->mask & IN_ATTRIB) != 0)
						printf("\tFile has changed attributes\n");
					if ((eventptr->mask & IN_CLOSE_WRITE) != 0)
						printf("\tFile was closed after write\n");
					if ((eventptr->mask & IN_CLOSE_NOWRITE) != 0)
						printf("\tFile was closed after read\n");
					if ((eventptr->mask & IN_CREATE) != 0)
						printf("\tNew file was created\n");
					if ((eventptr->mask & IN_DELETE) != 0)
						printf("\tExisting file was deleted\n");
					if ((eventptr->mask & IN_DELETE_SELF) != 0)
						printf("\tThe watched directory was deleted\n");
					if ((eventptr->mask & IN_MODIFY) != 0)
						printf("\tExisting file was modified\n");
					if ((eventptr->mask & IN_MOVE_SELF) != 0)
						printf("\tThe watched directory was moved\n");
					if ((eventptr->mask & IN_MOVED_FROM) != 0)
						printf("\tThe watched directory was moved (from) .. cookie = %u\n", eventptr->cookie);
					if ((eventptr->mask & IN_MOVED_TO) != 0)
						printf("\tThe watched directory was moved (to) .. cookie = %u\n", eventptr->cookie);
					if ((eventptr->mask & IN_OPEN) != 0)
						printf("\tThe file was opened\n");
					/* printf("wd=%d mask=%u cookie=%u len=%u\n", eventptr->wd, eventptr->mask, eventptr->cookie, eventptr->len); */

					pos += sizeof(struct inotify_event) + eventptr->len;
				}
			}
		}

		if (oneEvent == 1) {
			break;
		}
	}

	closeWatchDirs();
	free(buffer);
	return 0;
}

int main(int argc, char *argv[])
{
	const char *msg;

	uid = getuid();
	gid = getgid();

	msg = checkargs(argc, argv);
	if (msg != NULL) {
		usage(argv[0], msg);
		exit(1);
	}
	if (printHelp) {
		usage(argv[0], NULL);
		exit(0);
	}

	if (collectDirs(dirname) != 0) {
		exit(1);
	}

	if (watchDirs() != 0) {
		exit(1);
	}

	return 0;
}
