#ifndef _NOEXCLUSIONS
#include <exclude.h>	/* Define Preprocessor variables to */
			/* exclude un-wanted header files. */
#endif
#include <windows.h>
#include <tchar.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <time.h>
#include <limits.h>
#include <stdarg.h>

#define sleep Sleep

// #define PATH_MAX MAX_PATH

#define snprintf _snprintf
#define vsnprintf _vsnprintf 
#define setenv(a, b, c) _putenv_s(a, b)
#define unsetenv(a) _putenv_s(a, "")

SERVICE_STATUS srvStatus;
SERVICE_STATUS_HANDLE srvStatusHandle;

static char *default_output = "NUL";

static int no_of_segments = 3;
static int no_of_lines = 100000;
static char *logbasename = NULL;
static char *pipename = NULL;
static char *outputfile;
static char *workdir = NULL;
static TCHAR serviceName[128];
static char **cmd = NULL;
static char *cmdline = NULL;
static char *exefile = NULL;
static BOOL runAsDaemon = TRUE;
static int last_logno = 0;
static int first_logno = INT_MAX;
static BOOL verbose = FALSE;
static volatile BOOL terminate = FALSE;
static volatile BOOL restart = TRUE;		/* we restart by default */
static DWORD status = 0;
static HANDLE paip;
static volatile HANDLE childHandle;
static volatile int childpid = -1;
static volatile BOOL havechild = FALSE;
static HANDLE sllog = INVALID_HANDLE_VALUE;

/* prototypes */
char *strndup(char *str, size_t chars);
void usage(char *argv[], char *msg);
DWORD hprintf(HANDLE wrc, char *format, ...);
BOOL check_name(char *name, char **target, BOOL mustExist);
BOOL getopts(int argc, char *argv[]);
BOOL parseAndSetEnvironment(int argc, char *argv[], int *idx);
void UpdateStatus(int newStatus, int check);
VOID WINAPI serverCtrlHandler(DWORD srvControl);
void WINAPI serviceMain(DWORD argc, LPTSTR argv[]);
void print_config();
int getFileNumber(char *name);
BOOL getlognumber();
int start_cmd();
int process();
int scrolllogMain(int argc, char *argv[]);


char *strndup(char *str, size_t chars)
{
	char *buffer;
	size_t n;

	buffer = (char *) malloc(chars + 1);
	if (buffer) {
		for (n = 0; ((n < chars) && (str[n] != 0)) ; n++)
			buffer[n] = str[n];
		buffer[n] = 0;
	}

	return buffer;
}


void usage(char *argv[], char *msg)
{
	HANDLE hstdout = GetStdHandle(STD_OUTPUT_HANDLE);
	if (hstdout == INVALID_HANDLE_VALUE) return;

	if(msg != NULL) {
		hprintf(hstdout, "Error: %s\n\n", msg);
	}
	hprintf(hstdout, "Usage: \n");
	hprintf(hstdout, "%s [OPTIONS] name [-e cmdline]\n\n", argv[0]);
	hprintf(hstdout, "Options:\n");
	hprintf(hstdout, "-s NUMBER\tnumber of segments (default 3)\n");
	hprintf(hstdout, "-l NUMBER\tnumber of lines per segment (default 100.000)\n");
	hprintf(hstdout, "-c NAME\tname of the service\n");
	hprintf(hstdout, "-f\t\tRun in foreground (don't daemonize)\n");
	hprintf(hstdout, "-h\t\tDisplays this message\n");
	hprintf(hstdout, "-v\t\tDisplays options\n");
	hprintf(hstdout, "-o FILESPEC\tFile for own errormessages (only valid if daemon, default = /dev/null)\n\n");
	hprintf(hstdout, "-w WORKDIR\tWorking Directory, default = /\n");
	hprintf(hstdout, "-D VARIABLE=VALUE {VARIABLE=VALUE}\n");
	hprintf(hstdout, "name is the basename of the output files\n");
	hprintf(hstdout, "-e cmdline\tstarts the command with stdout and stderr redirected to the named pipe\n");
	hprintf(hstdout, "\t\t%s then terminates after the child process has terminated with exit code 0\n", argv[0]);
	hprintf(hstdout, "\t\tIf the child process terminates with exit code other than 0, it is restarted\n");
	hprintf(hstdout, "\t\tThe -e cmdline _MUST_ be the last option on the commandline. Everything following\n");
	hprintf(hstdout, "\t\tthe -e parameter is considered to be part of the commandline\n");
	hprintf(hstdout, "\t\tThe childprocess is started with the same environment as the calling process\n");
	hprintf(hstdout, "\t\tThe PATH environment variable is used\n");
	hprintf(hstdout, "\t\tWith the -D flag environment variables can be set\n");
}

DWORD hprintf(HANDLE wrc, char *format, ...)
{
#define BUFSIZE 2048
	va_list ap;
	char buf[BUFSIZE];
	DWORD nc;	/* number of chars */
	DWORD nw;	/* number written */

	if (wrc == INVALID_HANDLE_VALUE) return (DWORD) -1;
	va_start(ap, format);
	nc = vsnprintf(buf, BUFSIZE, format, ap);
	va_end(ap);
	if (nc < 0) return nc;
	if (nc == BUFSIZE) {
		/* would have been too long; we simply truncate it since it's only used for error messages */
		buf[BUFSIZE - 1] = '\0';
	}
	if (!WriteFile(wrc, buf, nc, &nw, NULL)) {
		/* TODO: process write error */;
	}
	if (nc != nw) {
		/* hm, what to do here ??? */
	}
	
	return nw;
}

BOOL check_name(char *name, char **target, BOOL mustExist)
{
	char *curdir = NULL;
	char *tmpdir = NULL;
	DWORD cwdlen;
	HANDLE srchHandle;
	WIN32_FIND_DATA findData;

/*
 * Step 1: replace all slashes by backslashes
 */
	curdir = name;
	while(*curdir) {
		if (*curdir == '/') *curdir = '\\';
		curdir++;
	}
	curdir = NULL;

	
/*
 * 	Absolute path if:
 * 	- <LETTER>:\\...
 * 	- \\<HOSTNAME>\...
 */
	if(!(isalpha(name[0]) && name[1] == ':' && name[2] == '\\') && !(name[0] == '\\' && name[1] == '\\')) {
		curdir = (char *) malloc(2 * PATH_MAX + 1);
		if (curdir == NULL)
			goto failure;
		tmpdir = (char *) malloc(2 * PATH_MAX + 1);
		if (tmpdir == NULL)
			goto failure;
		cwdlen = GetCurrentDirectory(PATH_MAX + 1, curdir);
		if(cwdlen == 0 || cwdlen > PATH_MAX + 1)
			goto failure;
		if (snprintf(curdir, 2 * PATH_MAX + 1, "%s\\%s", curdir, name) > 2 * PATH_MAX)
			goto failure;
		*target = curdir;
	} else {
		*target = name;
	}

/*
 * Now *target should be full qualified and point at something
 * That means, the path _must_exist
 */
	cwdlen = (DWORD) strlen(*target) - 1;
	while ((cwdlen > 0) && ((*target)[cwdlen] != '\\')) {
		cwdlen--;
	}
	if (tmpdir == NULL) tmpdir = (char *) malloc(cwdlen > 2 * PATH_MAX + 1 ? cwdlen : 2 * PATH_MAX + 1 );
	if (tmpdir == NULL) {
		goto failure;
	}
	strncpy(tmpdir, *target, cwdlen);
	tmpdir[cwdlen] = '\0';

	srchHandle = FindFirstFile(tmpdir, &findData);
	if (srchHandle == INVALID_HANDLE_VALUE) {
		int err = GetLastError();
		hprintf(sllog, "Got an invalid file handle (%d)\n", err);
		goto failure;
	}
	if ((findData.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) == 0) {
		hprintf(sllog, "The specified path isn't a directory\n");
		goto failure;
	}
	FindClose(srchHandle);
	
	if (mustExist) {
		srchHandle = FindFirstFile(*target, &findData);
		if (srchHandle == INVALID_HANDLE_VALUE) {
			hprintf(sllog, "%s should exist, but doesn't\n", *target);
			goto failure;
		}
	}
	FindClose(srchHandle);

	return TRUE;

failure:
	if (curdir != NULL)
		free(curdir);
	if (tmpdir != NULL)
		free(tmpdir);
	*target = NULL;
	return FALSE;
}


BOOL getopts(int argc, LPTSTR argv[])
{
	int i;
	int cmdsize;
	int cmdidx = 0;
	BOOL got_wd  = FALSE;
	BOOL got_sn  = FALSE;
	BOOL got_nos = FALSE;
	BOOL got_nol = FALSE;
	BOOL got_bn  = FALSE;
	BOOL got_of  = FALSE;
	char *endptr;

	for(i = 1; i < argc; i++) {
		if(!strcmp(argv[i], "-w")) {
			i++;
			if(i >= argc)	{ usage(argv, (char*) "Expected another parameter (workdir)"); return FALSE; }
			if(got_wd)	{ usage(argv, (char*) "Workdir specified twice"); return FALSE; }
			if(!check_name(argv[i], &workdir, TRUE))	{ usage(argv, (char*) "Invalid workdir"); return FALSE; }
			got_wd = TRUE;
		} else
		if(!strcmp(argv[i], "-c")) {
			i++;
			if(i >= argc)	{ usage(argv, (char*) "Expected another parameter (service name)"); return FALSE; }
			if(got_sn)	{ usage(argv, (char*) "Service name specified twice"); return FALSE; }
			if(serviceName[0] != '\0')	{ usage(argv, (char*) "Service name cannot be overridden"); return FALSE; }
			strncpy(serviceName, argv[i], 127);
			got_sn = TRUE;
		} else
		if(!strcmp(argv[i], "-s")) {
			i++;
			if(i >= argc)	{ usage(argv, (char*) "Expected another parameter (number of segments)"); return FALSE; }
			if(got_nos)	{ usage(argv, (char*) "Number of segments specified twice"); return FALSE; }
			no_of_segments = (int) strtol(argv[i], &endptr, 10);
			if((*endptr != '\0') || (no_of_segments < 0))	{ usage(argv, (char*) "Invalid numeric value for number of segments"); return FALSE; }
			got_nos = TRUE;
		} else
		if(!strcmp(argv[i], "-l")) {
			i++;
			if(i >= argc)	{ usage(argv, (char*) "Expected another parameter (number of lines per segment)"); return FALSE; }
			if(got_nol)	{ usage(argv, (char*) "Number of lines per segment specified twice"); return FALSE; }
			no_of_lines = (int) strtol(argv[i], &endptr, 10);
			if((*endptr != '\0') || (no_of_lines < 1))	{ usage(argv, (char*) "Invalid numeric value for number of lines per segment"); return FALSE; }
			got_nol = TRUE;
		} else
		if(!strcmp(argv[i], "-o")) {
			i++;
			if(i >= argc)	{ usage(argv, (char*) "Expected another parameter (outputfile)"); return FALSE; }
			if(got_of)	{ usage(argv, (char*) "Outputfile specified twice"); return FALSE; }
			if(!check_name(argv[i], &outputfile, FALSE))	{ usage(argv, (char*) "Invalid outputfilename"); return FALSE; }
			got_of = TRUE;
		} else
		if(!strcmp(argv[i], "-e")) {
			i++;
			if(i >= argc)	{ usage(argv, (char*) "Expected another parameter (commandline)"); return FALSE; }
			cmd = argv+i;
			cmdidx = i;
			i = argc;
		} else
		if(!strcmp(argv[i], "-h")) {
			usage(argv, NULL); return FALSE;
		} else
		if(!strcmp(argv[i], "-v")) {
			verbose = TRUE;
		} else
		if(!strcmp(argv[i], "-f")) {
			runAsDaemon = FALSE;
		} else
		if(!strcmp(argv[i], "-D")) {
			i++;
			if(i >= argc)	{ usage(argv, (char*) "Expected another parameter (environment variable specification)"); return FALSE; }
			if(!parseAndSetEnvironment(argc, argv, &i))	{ usage(argv, (char*) "Invalid environment specification"); return FALSE; }
		} else {
			if(got_bn)	{ usage(argv, (char*) "Output file name specified twice"); return FALSE; }
			logbasename = argv[i];
			got_bn = TRUE;
		}
	}
	/* check mandatory parameters */
	if (logbasename == NULL) { usage(argv, (char *) "basename for logfile is missing"); return FALSE; }
	if (workdir != NULL) SetCurrentDirectory(workdir);
	if (!check_name(logbasename, &logbasename, FALSE))	{ usage(argv, (char*) "Invalid output file name"); return FALSE; }
	if (no_of_segments < 2)	{ usage(argv, (char*) "Number of segments should be at least 2"); return FALSE; }
	if (runAsDaemon && serviceName == NULL) { usage(argv, (char*) "Specifying the service name is mandatory when running as service"); return FALSE; }

	sllog = CreateFile(outputfile == NULL ? default_output : outputfile, GENERIC_WRITE, FILE_SHARE_READ, NULL, CREATE_ALWAYS, FILE_ATTRIBUTE_NORMAL, NULL);
	if (sllog == INVALID_HANDLE_VALUE)
		return FALSE;
	SetStdHandle(STD_OUTPUT_HANDLE, sllog);
	SetStdHandle(STD_ERROR_HANDLE, sllog);

	cmdsize = 0;
	for (i = cmdidx; i < argc; ++i) {
		cmdsize += (DWORD) strlen(argv[i]) + 1 + 2;
	}
	cmdline = (char *) malloc(cmdsize + 1);
	if (cmdline == NULL) {
		hprintf(sllog, "Out of memory ... exiting\n");
		return FALSE;
	}

	cmdline[0] = '\0';
	for (i = cmdidx; i < argc; ++i) {
		strcat(cmdline, "\"");
		strcat(cmdline, argv[i]);
		strcat(cmdline, "\" ");
	}
	cmdline[strlen(cmdline) - 1] = '\0';	/* remove last space */
	exefile = strdup(argv[cmdidx]);
	if (exefile == NULL) {
		hprintf(sllog, "Out of memory ... exiting\n");
		return FALSE;
	}

	return TRUE;
}

BOOL parseAndSetEnvironment(int argc, char *argv[], int *idx)
{
	char *arg;
	int argptr = *idx;
	int arglen;
	int i;
	char *key;
	char *value;

	arg = argv[argptr];
	while (argptr < argc && arg[0] != '-') {
		/* arg should be of the form NAME=something */
		/* So we search for the first equals sign and split the string into a key and a value */
		arglen = strlen(arg);
		for (i = 0; i < arglen && arg[i] != '='; i++)
			/* just count */ ;
		if (i >= arglen) {
			return FALSE;
		}
		key = strndup(arg, i);
		if (i+1 >= arglen) {
			/* value is empty, we unset the valiable */
			value = NULL;
			unsetenv(key);
		} else {
			value = strdup((char *) &arg[i+1]);
			setenv(key, value, 1 /* overwrite */);
		}
		free(key);
		if (value != NULL) free(value);
		argptr++;
		if (argptr < argc) arg = argv[argptr];
	}
	*idx = argptr - 1;
	return TRUE;
}

void UpdateStatus (int newStatus, int check)
/*  Set a new service status and checkpoint (either specific value or increment) */
{
	if (check < 0 )	srvStatus.dwCheckPoint++;
	else		srvStatus.dwCheckPoint = check;
	if (newStatus >= 0) srvStatus.dwCurrentState = newStatus;
	if (runAsDaemon) {
		if (!SetServiceStatus (srvStatusHandle, &srvStatus) && (newStatus != SERVICE_STOPPED)) {
			hprintf (sllog, "Cannot set service status");
			srvStatus.dwCurrentState = SERVICE_STOPPED;
			srvStatus.dwWin32ExitCode = ERROR_SERVICE_SPECIFIC_ERROR;
			srvStatus.dwServiceSpecificExitCode = 2;
			UpdateStatus (SERVICE_STOPPED, -1);
			return;
		} 
	}

	return;
}

VOID WINAPI serverCtrlHandler(DWORD srvControl)
{
	switch (srvControl) {
		case SERVICE_CONTROL_SHUTDOWN:
		case SERVICE_CONTROL_STOP:
			terminate = TRUE;
			UpdateStatus (SERVICE_STOP_PENDING, -1);
			if (havechild) {
				HANDLE hProcess = OpenProcess(PROCESS_TERMINATE, FALSE, childpid);
				TerminateProcess(hProcess,0);
				CloseHandle(hProcess);
			}

			break;
		case SERVICE_CONTROL_PAUSE:
			/* not yet implemented */
			break;
		case SERVICE_CONTROL_CONTINUE:
			/* not yet implemented */
			break;
		case SERVICE_CONTROL_INTERROGATE:
			/* not yet implemented */
			break;
		default:
			/* User defined signals, not used; ignore */
			break;
	}
	UpdateStatus (-1, -1);
}

void WINAPI serviceMain(DWORD argc, LPTSTR argv[]) /* ARGSUSED */
{
	DWORD i;

	if (argc > 0) {
		for(i = 0; i < argc; ++i) {
			hprintf(sllog, "Parameter %d = %s\n", i, argv[i]);
		}
	} else {
		hprintf(sllog, "serviceMain is called without parameters\n");
	}
	srvStatus.dwServiceType = SERVICE_WIN32_OWN_PROCESS;
	srvStatus.dwCurrentState = SERVICE_START_PENDING;
	srvStatus.dwControlsAccepted = SERVICE_ACCEPT_STOP | SERVICE_ACCEPT_SHUTDOWN;
	srvStatus.dwWin32ExitCode = NO_ERROR;
	srvStatus.dwServiceSpecificExitCode = 0;
	srvStatus.dwCheckPoint = 1;
	srvStatus.dwWaitHint = 3000; /* TODO: what is this? */

	srvStatusHandle = RegisterServiceCtrlHandler (serviceName, serverCtrlHandler);
	hprintf(sllog, "Service registered\n");

	if (srvStatusHandle == 0) {
		hprintf(sllog, "Register Service went wrong ... stopping service\n");
		srvStatus.dwCurrentState = SERVICE_STOPPED;
		srvStatus.dwWin32ExitCode = ERROR_SERVICE_SPECIFIC_ERROR;
		srvStatus.dwServiceSpecificExitCode = 1;
		goto finish;
	}

	if (SetServiceStatus (srvStatusHandle, &srvStatus)) {
		hprintf(sllog, "Service status set, starting scrolllogMain()\n");
		scrolllogMain(argc, argv);
	} else {
		srvStatus.dwWin32ExitCode = ERROR_SERVICE_SPECIFIC_ERROR;
		srvStatus.dwServiceSpecificExitCode = 1;
	}

finish:
	/* in the process of shutting down (scrolllog terminated), we get here */
	UpdateStatus (SERVICE_STOPPED, 0);
	return;
}

void print_config()
{
	HANDLE hstdout = GetStdHandle(STD_OUTPUT_HANDLE);
	if (hstdout == INVALID_HANDLE_VALUE) return;

	hprintf(hstdout, "LogBase : %s\n", logbasename);
	hprintf(hstdout, "Working Directory : %s\n", workdir);
	hprintf(hstdout, "Scrolllog logfile : %s\n", outputfile);
	hprintf(hstdout, "Executable : %s\n", exefile);
	hprintf(hstdout, "Commandline : %s\n", cmdline);
	hprintf(hstdout, "Segments: %d\n", no_of_segments);
	hprintf(hstdout, "Lines   : %d\n", no_of_lines);
	hprintf(hstdout, "Last    : %d\n", last_logno);
	hprintf(hstdout, "First   : %d\n", first_logno);
}

int getFileNumber(char *name)
{
	char *nmbr;
	int  rc = -1;

	nmbr = &(name[strlen(name) - 1]);
	while (isdigit(*nmbr)) nmbr--;
	nmbr++;		/* the last character wasn't a digit */
	if (!isdigit(*nmbr)) return rc;	/* no number found */
	if ((strlen(name) - (nmbr - name)) > 9) return rc;	/* value out of range (> 999.999.999) */

	rc = (int) strtol(nmbr, NULL, 10);	/* due to construction of nmbr, it must be a number */

	return rc;
}

BOOL getlognumber()
{
	char *pattern;
	HANDLE ffHandle = INVALID_HANDLE_VALUE;
	WIN32_FIND_DATA findData;
	int minNo, maxNo, numFiles, tmpNo;

	pattern = (char *) malloc(strlen(logbasename) + 3);
	if (pattern == NULL) {
		hprintf(sllog, "Error allocating memory\n");
		goto gln_failure;
	}

	strcpy(pattern, logbasename);
	strcat(pattern, ".*");

	ffHandle = FindFirstFile(pattern, &findData);
	if (ffHandle == INVALID_HANDLE_VALUE) {
		if (GetLastError() != ERROR_FILE_NOT_FOUND) {
			hprintf(sllog, "last error = %x\n", GetLastError());
			goto gln_failure;
		}
		minNo = 0; maxNo = 0; numFiles = 0;
	} else {
		numFiles = 1;
		minNo = maxNo = getFileNumber(findData.cFileName);
		while (FindNextFile(ffHandle, &findData)) {
			numFiles++;
			tmpNo = getFileNumber(findData.cFileName);
			if (tmpNo < 0) continue; /* no number found, not one of ours */
			if (tmpNo < minNo) minNo = tmpNo;
			if (tmpNo > maxNo) maxNo = tmpNo;
		}
	}

	last_logno = maxNo;
	first_logno = minNo;

	FindClose(ffHandle);
	free(pattern);
	
	return TRUE;

gln_failure:
	if (pattern != NULL) free(pattern);
	if (ffHandle != INVALID_HANDLE_VALUE) FindClose(ffHandle);
	return FALSE;
}

int start_cmd()
{
	HANDLE outp;
	SECURITY_ATTRIBUTES pipeSA = {sizeof(SECURITY_ATTRIBUTES), NULL, TRUE};
	PROCESS_INFORMATION pi;
	STARTUPINFO si;

	if (!CreatePipe (&paip, &outp, &pipeSA, 0)) {
		hprintf(sllog, "Error creating pipe : %x\n", GetLastError());
		return 1;
	}

	/* don't inherit (our) input handle */
	SetHandleInformation(paip, HANDLE_FLAG_INHERIT, 0);

	/* set stdout and stderr to the other end of the pipe (because of inheritance) */
	/*
	SetStdHandle(STD_OUTPUT_HANDLE, outp);
	SetStdHandle(STD_ERROR_HANDLE, outp);
	*/

	/* start the child process */
	ZeroMemory( &pi, sizeof(PROCESS_INFORMATION) );
	ZeroMemory( &si, sizeof(STARTUPINFO) );
	si.cb = sizeof(STARTUPINFO); 
	si.hStdError = outp;
	si.hStdOutput = outp;
	si.hStdInput = GetStdHandle(STD_INPUT_HANDLE);
	si.dwFlags |= STARTF_USESTDHANDLES;

	if(!CreateProcess(NULL,		/* command, not used */
			  cmdline,	/* the commandline as specified */
			  NULL,		/* process security attributes */
			  NULL,		/* primary thread security attributes */
			  TRUE,		/* inherit handles */
			  0,		/* creation flags */
			  NULL,		/* use parent's environment */
			  NULL,		/* use parent's working directory */
			  &si,		/* startup info */
			  &pi)) {	/* process info */
		hprintf(sllog, "Create Process failed : %x\n", GetLastError());
		return 1;
	}

	/* get the pid and other information */
	CloseHandle(pi.hThread);	/* we don't need this handle, therefore we free it */
	childHandle = pi.hProcess;
	childpid = pi.dwProcessId;
	havechild = TRUE;

	/* now close the handle for stdout and stderr */
	CloseHandle(outp);

	/* restore our error channel */
	SetStdHandle(STD_ERROR_HANDLE, sllog);

	return 0;
}

int process()
{
#define BUFSIZE 2048
	HANDLE outf;
	char filename[PATH_MAX + 1];
	int lineno;
	DWORD numRead;
	DWORD numChars;
	static char buf[BUFSIZE+1];
	int myTerminate = 0;

	while(myTerminate != 2 && !terminate) {
		/* delete oldest file(s) if necessary */
		/* we assume that all files are present (no missing numbers) */
		/* if not: Shit happens */
		while(last_logno - first_logno > no_of_segments) {
			snprintf(filename, PATH_MAX, "%s.%d", logbasename, first_logno);
			DeleteFile(filename);	/* we accept errors like missing filenames or access violations */
			first_logno++;
		}
		/*     open next file */
		last_logno++;
		snprintf(filename, PATH_MAX, "%s.%d", logbasename, last_logno);

		outf = CreateFile(filename, GENERIC_WRITE, FILE_SHARE_READ, NULL, CREATE_NEW, FILE_ATTRIBUTE_NORMAL, NULL);
		if(outf == INVALID_HANDLE_VALUE) {
			hprintf(sllog, "Error opening file %s : %s\n", filename, strerror(errno));
			if(havechild) {
				HANDLE hProcess = OpenProcess(PROCESS_TERMINATE, FALSE, childpid);
				TerminateProcess(hProcess,0);
				CloseHandle(hProcess);
				sleep(10);	
			}
			return 9;
		}

		lineno = 0;
		while((lineno < no_of_lines) && (myTerminate != 2) && !terminate) {
			UpdateStatus(-1, -1);
			if (!ReadFile(paip, buf, BUFSIZE, &numRead, NULL)) {
				/* ReadFile returned FALSE. This means either a read error or EOF */
				if (numRead == 0) {	/* EOF */
					CloseHandle(paip);
					FlushFileBuffers(outf);	/* just in case */
					if (havechild) {	/* did we execute a command in the first place ? */
						hprintf(outf, "[scrolllog] Waiting for child (%d) to terminate\n", (int) childpid);
						FlushFileBuffers(outf);
						WaitForSingleObject(childHandle, INFINITE);
						GetExitCodeProcess(childHandle, &status);
						numChars = snprintf(buf, BUFSIZE, "[scrolllog] Child exited with state %d\n", status);
						if(!WriteFile(outf, buf, numChars, &numChars, NULL)) {
							/* TODO: process write errors */
						}
						FlushFileBuffers(outf);
						CloseHandle(childHandle);
						childpid  = -1;
						havechild = FALSE;
						myTerminate = 1;
						if (status == 0) restart = 0; else restart = 1;
					}
				} else {
					/* TODO process Read Error */
					hprintf(sllog, "Reading from pipe ... returned FALSE\n");
				}
				if(myTerminate == 1) {
					if(restart && (cmd != NULL) && !terminate) {
						hprintf(outf, "[scrolllog] Try to restart child (child terminated with exit code <> 0)\n");
						FlushFileBuffers(outf);
						if(start_cmd() != 0) {
							hprintf(outf, "[scrolllog] restart of child failed\n");
							FlushFileBuffers(outf);
							exit(9);
						}
						myTerminate = 0;
					} else {	
						myTerminate = 2;	/* start termination phase 2 (and leave inner loop) */
						hprintf(outf, "[scrolllog] I terminate (after cleaning up)\n");
						FlushFileBuffers(outf);
						break;
					}
				} 
	
				continue;
			}
			WriteFile(outf, buf, numRead, &numRead, NULL);
			FlushFileBuffers(outf);
			lineno++;	/* must not be right, but is OK for normal purpose */
		}
		CloseHandle(outf);
	}

	return status;
}

int scrolllogMain(int argc, LPTSTR argv[])
{
	UpdateStatus (-1, -1);
	UpdateStatus (SERVICE_RUNNING, -1);
	hprintf(sllog, "Status set to RUNNING\n");

	if (argc > 1)	/* don't evaluate if we only get a program name */ 
		if (!getopts(argc, argv)) return 1;

	if(!getlognumber()) {
		hprintf(sllog, "Cannot determine actual log number\n");
		return 1;
	}
	if(verbose) {
		print_config();
	}

	if(cmd != NULL) {
		if(start_cmd() != 0) exit(9);
		hprintf(sllog, "Command started\n");
	}

	return process();
}

int _tmain(int argc, LPTSTR argv[])
{
	SERVICE_TABLE_ENTRY dispatchTable[2];

	sllog = GetStdHandle(STD_ERROR_HANDLE);
	if (!getopts(argc, argv)) return 1;

	/* compiler compained about using variables in a static initialization
	   that's why it is done "by hand"
	*/
	dispatchTable[0].lpServiceName = serviceName;
	dispatchTable[0].lpServiceProc = serviceMain;
	dispatchTable[1].lpServiceName = NULL;
	dispatchTable[1].lpServiceProc = NULL;

	if (runAsDaemon) {
		hprintf(sllog, "calling StartServiceCtrlDispatcher with parameters serviceName (%s) and scrollogSrvMain (%X)\n", dispatchTable[0].lpServiceName, dispatchTable[0].lpServiceName);
		if (!StartServiceCtrlDispatcher(dispatchTable))
			hprintf(sllog, "Error starting Service Control Dispatcher : %d\n", GetLastError());
	} else {
		/* run in foreground */
		return scrolllogMain(0, (char **) NULL);
	}
	return 0;
}
