﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Text;

namespace Scrolllog
{
	public class ExitEventArgs : EventArgs
	{
		public String ServiceName { get; set; }
	}

	public class Scrolllog
	{
		private Process _p;
		private int _exitCode             = -1;
		private int _numberOfSegments     = 3;
		private int _numberOfLines        = 100000;
		private string _executable        = "";
		private string _basename          = "";
		private string _workdir           = @"C:\Temp";
		private string _outfiledir        = @"";
		private string _logdir            = @"";
		private string _commandlineParams = "";
		private bool _verbose             = false;
		private string _outfilename       = "";
		private int _currentLine          = 0;
		private int _currentFile          = 1;
		private TextWriter _logwriter;
		private String _serviceName       = ""; // only required when started as Service
		private bool _runAsService        = false;
		private bool _run                 = true;
		
		private const int INITIAL_CODE         = -1;
		private const int EXECUTABLE_NOT_FOUND = -5;
		private const int EXEC_EXCEPTION       = -10;

		public event EventHandler<ExitEventArgs> ExitServiceEvent;
		
		public Scrolllog(String[] args)
		{
			_run = GetOps(args);
		}

		public void ThrowExitServiceEvent(Object sender, ExitEventArgs e)
		{
			if (ExitServiceEvent != null)
				ExitServiceEvent(sender, e);
		}
		
		public bool runsAsService() { return _runAsService; }
		public string getServiceName() { return _serviceName; }

		/// <summary>
		/// Method to start the Scrolllog as Commandline Tool
		/// </summary>
		/// <param name="args"></param>
		public void RunScrolllog()
		{
			// _run is set to false if there was a problem with the commandline (or -h was specified)
			if (!_run) {
				if (_runAsService) {					
					ExitEventArgs e = new ExitEventArgs();
					e.ServiceName = _serviceName;
					ThrowExitServiceEvent(this, e);
				}
				return;
			}
			
			int exitcode = INITIAL_CODE;

			if (_verbose && !_runAsService) {
				PrintConfig();
			}

			//run the loop as long as the process hasn´t exited with status 0
			while (exitcode != 0) {
				exitcode = StartWatching();

				if (exitcode == EXECUTABLE_NOT_FOUND || exitcode == EXEC_EXCEPTION) {
					if (_runAsService) {
						ExitEventArgs e = new ExitEventArgs();
						e.ServiceName = _serviceName;
						ThrowExitServiceEvent(this, e);
					}
					break;
				} else if (exitcode > 0) {
					string msg = "Process terminated with Exitcode " + exitcode + ", trying to restart it!";
					WriteLog(msg);
				}
			}
			
			if (_runAsService) {
				ExitEventArgs ev = new ExitEventArgs();
				ev.ServiceName = _serviceName;
			
				ThrowExitServiceEvent(this, ev);
			}
		}


		/// <summary>
		/// Read and assign Program parameters
		/// </summary>
		/// <param name="args"></param>
		/// <returns></returns>
		public bool GetOps(String[] args)
		{
			int argc = args.Count();
			bool got_numbersOfLines = false;
			bool got_numbersOfSegments = false;
			bool got_basename = false;
			bool got_workdir = false;
			bool got_commandline = false;
			bool got_outfilename = false;
			bool got_serviceName = false;
			bool got_dont_ras = false;

			
			// Quit if no parameters are specified
			if (argc == 0) {
				Usage("");
				return false;
			}

			for (int i = 0; i < argc && !got_commandline; i++) {
				if (String.Compare(args[i], "-w") == 0)	{
					i++;
					if (!checkDups(i, argc, got_workdir, "workdir")) return false;
					if (!checkFilespec(args[i], false, true, ref _workdir, ref _workdir)) return false;

					got_workdir = true;

				} else if (String.Compare(args[i], "-c") == 0) {
					i++;
					if (!checkDups(i, argc, got_serviceName, "service name")) return false;
					// if someone attempts to specify another service name, don't start
					// the _serviceName is initialized with an empty string therefore it cannot 
					// be filled here, unless it was already specified in the previous evaluation
					// of arguments
					if (!Utility.IsNullOrEmptyOrBlank(_serviceName)) return false;
					_serviceName = args[i];
					
					if (Utility.IsNullOrEmptyOrBlank(_serviceName) || _serviceName.Length > 80 ||
					    !Utility.ValidFilePath(_serviceName) || !Utility.ValidFileName(_serviceName)
					   ) {
						Usage("Invalid Service name.");
						return false;
					}
					
					got_serviceName = true;

				} else if (String.Compare(args[i], "-s") == 0) {
					i++;
					if (!checkNumeric(i, argc, args, ref got_numbersOfSegments, out _numberOfSegments, "number of segments"))
						return false;
				} else if (String.Compare(args[i], "-f") == 0) {
					// _runAsService is initialized with false
					// if it has become true, someone is trying to run in foreground
					// while editing start parameters in the service dialogue
					if (_runAsService) return false;
					got_dont_ras = true;
				} else if (String.Compare(args[i], "-o") == 0) {
					i++;
					if (!checkDups(i, argc, got_outfilename, "outputfile")) return false;
					if (!checkFilespec(args[i], true, false, ref _outfiledir, ref _outfilename)) return false;

					//Errorlog file is created even if there will be no errors to log
					WriteError(new Exception("Scrollog starts ..."));
					got_outfilename = true;
				} else if (String.Compare(args[i], "-l") == 0) {
					i++;
					if (!checkNumeric(i, argc, args, ref got_numbersOfLines, out _numberOfLines, "number of lines"))
						return false;
				} else if (String.Compare(args[i], "-e") == 0) {
					i++;
					if (!checkDups(i, argc, got_commandline, "command line")) return false;

					//Build the command with all the passed params
					StringBuilder strb = new StringBuilder();

					_executable = args[i].Replace("/", "\\");
					if (_executable.Contains("\\")) {
						if (_executable.Contains("\""))	{
							Usage("Invalid executable name (must not contain \\\").");
							return false;
						}
					} else {
						if (!Utility.ValidFileName(_executable)) {
							Usage("Invalid executable name.");
							return false;
						}
					}
					
					for (int y = i + 1; y < argc; ++y)
						strb.Append("\"" + args[y] + "\" ");

					_commandlineParams = strb.ToString().TrimEnd();
					got_commandline = true;

				} else if (String.Compare(args[i], "-h") == 0) {
					Usage("");
					return false;
				} else if (String.Compare(args[i], "-v") == 0)
					_verbose = true;
				else {
					if (!checkDups(i, argc, got_basename, "name")) return false;
					if (!checkFilespec(args[i], true, false, ref _logdir, ref _basename)) return false;
					got_basename = true;
				}
			}

			_currentFile = GetMaxLognumber() + 1;
			_runAsService = !got_dont_ras;
			
			if (_runAsService && Utility.IsNullOrEmptyOrBlank(_serviceName)) {
				Usage("Service name is obligatory if scrolllog is run as a service");
				return false;
			}
			if (got_dont_ras && !Utility.IsNullOrEmptyOrBlank(_serviceName)) {
				Usage("Service name specified although not running as service");
				return false;
			}
			if (!got_commandline) {
				Usage("Command line is missing");
				return false;
			}
			if (!got_basename) {
				Usage("name parameter is missing");
				return false;
			}

			return true;
		}
		
		private bool checkFilespec(string name, bool canWrite, bool pathOnly, ref string resultpath, ref string resultfname)
		{
			if (Utility.IsNullOrEmptyOrBlank(name)) {
				Usage("File or path argument is empty");
				return false;
			}
			
			string fullname = name.Replace("/", "\\");
			string path = fullname;

			if (fullname.Contains("\\") && !pathOnly) {
				path = fullname.Substring(0, name.LastIndexOf("\\") + 1);
				fullname = fullname.Substring(name.LastIndexOf("\\") + 1);
			} else {
				if (!pathOnly) {
					path = Environment.CurrentDirectory;
				} else {
					fullname = "";
				}
			}
			
			if (path.Contains("\"")) {
				Usage("Invalid filename (must not contain \\\").");
				return false;
			}
			if (!Utility.ValidFilePath(path)) {
				Usage("Invalid path.");
				return false;
			}
			if (!Directory.Exists(path))	{
				Usage("The specified directory (" + path + ") does not exist.");
				return false;
			}
			if (canWrite && !isWritable(path)) {
				Usage("No permissions to write " + path + ".");
				return false;
			}
			if (!pathOnly) {
				if (!Utility.ValidFileName(fullname)) {
					Usage("Invalid file name: '" + fullname + "'.");
					return false;
				}
				if (Utility.IsNullOrEmptyOrBlank(fullname)) {
					Usage("Empty file name.");
					return false;
				}
			}
			
			if (path != "" && resultpath != null)
				resultpath = Path.GetFullPath(path);
			if (!pathOnly)
				resultfname = fullname;
			
			return true;
		}
		
		private bool checkDups(int i, int argc, bool got_item, string name)
		{
			if (i > argc) {
				Usage("Expected another parameter (" + name + ").");
				return false;
			}

			if (got_item)	{
				Usage(name + " specified twice.");
				return false;
			}
			return true;
		}
		
		private bool checkNumeric(int i, int argc, string[] args, ref bool got_item, out int numVal, string name)
		{
			if (!checkDups(i, argc, got_item, name)) {
				numVal = 0;
				return false;
			}

			if (!Int32.TryParse(args[i], out numVal)) {
				Usage("Invalid numeric value for " + name + ".");
				got_item = false;
				numVal = 0;
				return false;
			}

			if (numVal <= 0) {
				Usage(name + " <= 0.");
				got_item = false;
				numVal = 0;
				return false;
			}

			got_item = true;
			return true;
		}

		private int StartWatching()
		{
			string executablePath = "";
			// check if executable exists in %Path%
			if (!existsExecutable(ref _executable, ref executablePath)) {
				Usage("Executable file " + _executable + " not found.");
				return EXECUTABLE_NOT_FOUND;
			}

			ProcessStartInfo psi = new ProcessStartInfo();
			psi.FileName = executablePath;

			//now start the execution
			psi.WindowStyle = ProcessWindowStyle.Maximized;
			psi.RedirectStandardError = true;
			psi.RedirectStandardOutput = true;
			psi.UseShellExecute = false;
			psi.CreateNoWindow = true;
			psi.WorkingDirectory = _workdir;

			if (!string.IsNullOrWhiteSpace(_commandlineParams)) {
				psi.Arguments = _commandlineParams;
			}
			
			_p = new Process();
			_p.StartInfo = psi;
			_p.EnableRaisingEvents = true;
			_p.ErrorDataReceived += _p_ErrorDataReceived;
			_p.OutputDataReceived += _p_OutputDataReceived;
			
			try {
				_p.Start();
				_p.BeginErrorReadLine();
				_p.BeginOutputReadLine();
				_p.WaitForExit();
				_exitCode = _p.ExitCode;
				return _exitCode;

			} catch (Win32Exception w32ex) {
				_exitCode = EXECUTABLE_NOT_FOUND;
				WriteError(w32ex);
				return _exitCode;
			} catch (Exception ex) {
				WriteError(ex);
				_exitCode = EXEC_EXCEPTION;
				return _exitCode;
			}
		}

		/// <summary>
		/// Method that checks if executable is in %Path% of the current (parent) process
		/// </summary>
		private bool existsExecutable(ref string filename, ref string qualPath)
		{
			string pathEnvironment = "";
			string pathVal = "";
			string pathExt = null;
			
			try {
				pathExt = System.Environment.GetEnvironmentVariable("PATHEXT");
			} catch (System.Security.SecurityException se) {
				WriteError(se);
				return false;
			} catch (ArgumentNullException) {
				pathExt = null;
			}
			string[] extentions = null;
			if (pathExt != null)
				extentions = pathExt.Split(";".ToCharArray());

			//relative path?
			if (!Path.IsPathRooted(filename)) {
				//consider working dir if relative path of executable is given

				if (filename.Contains("\\") || filename.Contains("/")) {
					//check workdir
					pathVal = Path.GetFullPath(Path.Combine(_workdir, filename));
					qualPath = fileExists(pathVal, extentions);
					if (qualPath != null) return true;
				} else {

					//check $PATH
					try {
						pathEnvironment = System.Environment.GetEnvironmentVariable("PATH");
					} catch (System.Security.SecurityException se) {
						WriteError(se);
						return false;
					} catch (ArgumentNullException ans) {
						WriteError(ans);
						return false;
					}

					string[] pathA = pathEnvironment.Split(";".ToCharArray());

					foreach (string s in pathA) {
						try {
							pathVal = Path.GetFullPath(Path.Combine(s, filename));
							qualPath = fileExists(pathVal, extentions);
							if (qualPath != null) return true;
						} catch (ArgumentException ae) {
							WriteError(ae);
							return false;
						}
					}
				}

			} else {
				qualPath = fileExists(filename, extentions);
				if (qualPath != null) return true;
			}
			
			return false;
		}
		
		private string fileExists(string name, string[] extentions)
		{
			if (File.Exists(name)) {
				return name;
			}
			if (extentions == null) return null;
			foreach (string ext in extentions) {
				if (File.Exists(name + ext)) {
					return name + ext;
				}
			}
			return null;
		}

		/// <summary>
		/// Method to print out the configuration
		/// </summary>
		private void Usage(String Message)
		{
			if (!string.IsNullOrWhiteSpace(Message)) {
				WriteError(new Exception(Message));
				Console.Out.WriteLine("\nError: " + Message + "\n\n");
			}

			Console.Write("Usage: \n");
			Console.Write("{0} [OPTIONS] name [-e cmdline]\n\n", "Scrolllog");
			Console.Write("Options:\n");
			Console.Write("-s NUMBER\tnumber of segments (default 3)\n");
			Console.Write("-l NUMBER\tnumber of lines per segment (default 100.000)\n");
			Console.Write("-c service name required if Scrolllog is run as service (must be equal to the registered Scrolllog service name)\n");
			Console.Write("-h Displays this message\n");
			Console.Write("-f Run in foreground\n");
			Console.Write("-v Displays the effective options if running in foreground\n");
			Console.Write("-o FILESPEC\tFile for own errormessages \n");
			Console.Write("-w WORKDIR\tWorking Directory, default = C:\\Temp\n");
			Console.Write("name is the basename of the logfiles\n");
			Console.Write("-e cmdline\tstarts the command with stdout and stderr redirected\n");
			Console.Write("\t\tto the scrolllog process.\n");
			Console.Write("\t\tScrolllog then terminates after the child process has terminated\n");
			Console.Write("\t\twith exit code 0\n");
			Console.Write("\t\tIf the child process terminates with exit code other than 0,\n"); 
			Console.Write("\t\tit is restarted\n");
			Console.Write("\t\tThe -e cmdline _MUST_ be the last option on the commandline.\n");
			Console.Write("\t\tEverything following the -e parameter is considered to be part\n");
			Console.Write("\t\tof the commandline\n");
			Console.Write("\t\tThe childprocess is started with the same environment as the\n");
			Console.Write("\t\tcalling process. The PATH environment variable is used\n");
			
			Console.ReadLine();
		}

		/// <summary>
		/// Output of the current configuration
		/// </summary>
		private void PrintConfig()
		{
			Console.Out.WriteLine("Name              : {0}", System.Diagnostics.Process.GetCurrentProcess().ProcessName);
			Console.Out.WriteLine("LogBase           : {0}", Path.Combine(_logdir, _basename).ToString());
			Console.Out.WriteLine("Working Directory : {0}", _workdir);
			Console.Out.WriteLine("Scrolllog logfile : {0}", Path.Combine(_outfiledir, _outfilename).ToString());
			Console.Out.WriteLine("Executable        : {0}", _executable);
			Console.Out.WriteLine("Command line parms: {0}", _commandlineParams);
			Console.Out.WriteLine("Segments          : {0}", _numberOfSegments);
			Console.Out.WriteLine("Lines             : {0}", _numberOfLines);
			Console.Out.WriteLine("Last              : {0}", GetMaxLognumber());
			Console.Out.WriteLine("First             : {0}", GetLowestLognumber());
			Console.ReadLine();
		}

		/// <summary>
		/// Event-Handler for incoming Stdout-Messages
		/// Prints Messages to the files specified in the class variables
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		void _p_OutputDataReceived(object sender, DataReceivedEventArgs e)
		{
			if (!string.IsNullOrEmpty(e.Data)) {
				WriteLog(e.Data);
			}
		}

		/// <summary>
		/// Event-Handler for incoming Stderr-Messages
		/// Prints Messages to the files specified in the class variables
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		void _p_ErrorDataReceived(object sender, DataReceivedEventArgs e)
		{
			if (!string.IsNullOrEmpty(e.Data)) {
				WriteLog(e.Data);
			}
		}

		void WriteLog(string message)
		{
			string FileName = "";

			try {
				if (_currentLine >= _numberOfLines) {
					//Neue Datei anlegen und bei 0 beginnen
					_currentFile++;
					_currentLine = 0;
					FileName = Path.Combine(_logdir, _basename + "." + _currentFile);

					_logwriter.Close();

					_logwriter = File.CreateText(FileName);

					//Löschen der ältesten Datei
					while(numberOfFilesWithBaseName(_logdir) > _numberOfSegments) {
						File.Delete(Path.Combine(_logdir, _basename + "." + GetLowestLognumber()));
					}
				} else {
					FileName = Path.Combine(_logdir, _basename + "." + _currentFile);

					if (_logwriter == null)
						_logwriter = File.CreateText(FileName);
				}

				lock (_logwriter) {
					_logwriter.WriteLine(message);
					_logwriter.Flush();
				}
				
				_currentLine++;
			} catch (Exception ex) {
				WriteError(ex);
			}

		}

		/// <summary>
		/// Holt die Nummer des aktuellen Logfiles!
		/// </summary>
		/// <returns></returns>
		private int GetMaxLognumber()
		{
			return GetFileNumbersList().Max();

		}

		/// <summary>
		/// Holt die Nummer das aelteste Logfile
		/// </summary>
		/// <returns></returns>
		private int GetLowestLognumber()
		{
			return GetFileNumbersList().Min();
		}

		/// <summary>
		/// Holt eine Liste mit den aktuellen Filenumbers
		/// </summary>
		/// <returns></returns>
		private List<int> GetFileNumbersList()
		{
			string searchStr = _basename + ".*";
			FileInfo[] files = new DirectoryInfo(_logdir).GetFiles(searchStr);

			//consider only those files that end with an int
			List<string> selectedfilenames = new List<string>();

			foreach (FileInfo fi in files) {
				string ex = fi.Name;
				if (ex.Contains(_basename)) {
					string end = ex.Substring(_basename.Length); //+1 für den Punk vor der zu erwartenden Nummer
					if (end.StartsWith(".")) {
						end = end.Substring(1);
					}
					int filenumber = 0;
					if (int.TryParse(end, out filenumber)) {
						selectedfilenames.Add(fi.Name);
					}
				}

			}

			List<int> numbers = new List<int>();

			if(selectedfilenames.Count == 0) {
				numbers.Add(0);
				return numbers;
			}

			selectedfilenames.ForEach((string m) =>
			                          {
			                          	numbers.Add(Utility.GetNumberFromString(m));
			                          });
			return numbers;
		}

		/// <summary>
		/// Method to write an own error
		/// </summary>
		/// <param name="ex"></param>
		public void WriteError(Exception ex)
		{
			if (string.IsNullOrEmpty(_outfilename))
				return;

			try {
				FileInfo fi = new FileInfo(Path.Combine(_outfiledir, _outfilename));
				using (System.IO.StreamWriter file = new System.IO.StreamWriter(fi.FullName, true))
				{
					lock (file) {
						file.WriteLine("--------- begin of error message at: " + DateTime.Now.ToString() + " ------------------");

						file.WriteLine(ex.Message.ToString());

						Exception innerEx = ex.InnerException;

						while (innerEx != null)	{
							file.WriteLine(" *************** ex.InnerException.Message ********************");
							file.WriteLine(innerEx.Message.ToString());
							innerEx = innerEx.InnerException;
						}

						file.WriteLine("--------------------- end of error message -------------------------");
					}
				}
			} catch (Exception) {
				// ignore
			}
		}

		public int numberOfFilesWithBaseName(string directory)
		{
			string searchStr = _basename + ".*";
			FileInfo[] files = new DirectoryInfo(directory).GetFiles(searchStr);

			//consider only those files that end with an int
			List<string> selectedfilenames = new List<string>();

			foreach (FileInfo fi in files) {
				string ex = fi.Name;
				if (ex.Contains(_basename)) {
					string end = ex.Substring(_basename.Length); //+1 für den Punkt vor der zu erwartenden Nummer
					if (end.StartsWith(".")) {
						end = end.Substring(1);
					}
					int filenumber = 0;
					if (int.TryParse(end, out filenumber)) {
						selectedfilenames.Add(fi.Name);
					}
				}
			}

			return selectedfilenames.Count();
		}


		public bool isWritable(string directory)
		{
			string FileName = "";
			TextWriter writer = null;

			try {
				FileName = Path.Combine(directory, "test.txt");
				writer = File.CreateText(FileName);
				writer.Close();
				File.Delete(FileName);
			} catch (UnauthorizedAccessException)	{
				//Usage(" No permissions for writing directory " + _logdir);
				//Console.Out.WriteLine(" No permissions for writing directory " + directory);
				return false;
			} catch (Exception) {
				//Usage(ex.Message);
				return false;
			} finally	{
				if (writer != null)
					writer.Dispose();
			}

			return true;
		}

		public void ScrolllogStop()
		{
			if (!_p.HasExited) {
				string error = _p.StandardError.ReadToEnd();
				WriteLog(error);
				_p.WaitForExit(1000);
				_p.Close();

				_logwriter.Dispose();
			}
		}
		
	}
}
