/* 
// simplesc is a small utility to manage services
// This utility isn't strictly needed on newer Windows systems since 
// those are equiped with sc.exe. simplesc is nothing more than a
// utility copying the functionality of sc.exe
//
// Copyright (c) 2014 independIT Integrative Technologies GmbH. All rights reserved.
*/

#include <windows.h>
#include <tchar.h>
#include <stdio.h>

#define NO_ACTION  0
#define CREATE     1
#define REMOVE     2
#define START      3
#define STOP       4
#define QUERY      5
#define CONFIG     6
#define HELP       7

TCHAR *serviceName;
TCHAR *path;
DWORD startMode;
DWORD errcd;

/* Prototypes */
BOOL PrintUsage();
BOOL InstallService(TCHAR *serviceName, DWORD startMode, TCHAR *path);
BOOL UninstallService(TCHAR *serviceName);
BOOL ConfigService(TCHAR *serviceName, DWORD startMode, TCHAR *path);
BOOL QueryService(TCHAR *serviceName);
BOOL StartupService(TCHAR *serviceName);
BOOL StopService(TCHAR *serviceName);
BOOL parseArgs(int argc, TCHAR* argv[], int *action);


BOOL PrintUsage()
{
	fprintf(stderr, "Usage: simplesc <operation> <servicename> {options}\n");
	fprintf(stderr, "\n");
	fprintf(stderr, "<operation> is one of\n");
	fprintf(stderr, "\tcreate, config, delete, query, start, stop\n");
	fprintf(stderr, "\n");
	fprintf(stderr, "Depending on the operation, following options are recognized:\n");
	fprintf(stderr, "\tstart= <startmode>\n\t\t<startmode> is either \"auto\" or \"manual\"\n");
	fprintf(stderr, "\tbinpath= \"<commandline>\"\n\t\tNOTE: the commandline must be enclosed in \"\"\n\n");
	return TRUE;
}

BOOL InstallService(TCHAR *serviceName, DWORD startMode, TCHAR *path)
{
	BOOL rc = TRUE;

	SC_HANDLE serviceControlManager = OpenSCManager( 0, 0, SC_MANAGER_CREATE_SERVICE );

	if (serviceControlManager != NULL) {
		SC_HANDLE service = CreateService(
			serviceControlManager,
			serviceName, serviceName,
			SERVICE_ALL_ACCESS, SERVICE_WIN32_OWN_PROCESS,
			startMode, SERVICE_ERROR_IGNORE, path,
			0, 0, 0, 0, 0 );
		if ( service != NULL)
			CloseServiceHandle( service );
		else {
			rc = FALSE;
			errcd = GetLastError();
		}
		CloseServiceHandle( serviceControlManager );
	} else {
		rc = FALSE;
		errcd = GetLastError();
	}
	return rc;
}

BOOL UninstallService(TCHAR *serviceName)
{
	BOOL rc = TRUE;
	SERVICE_STATUS serviceStatus;

	SC_HANDLE serviceControlManager = OpenSCManager( 0, 0, SC_MANAGER_CONNECT );

	if (serviceControlManager != NULL) {
		SC_HANDLE service = OpenService( serviceControlManager, serviceName, SERVICE_QUERY_STATUS | DELETE );
		if (service != NULL) {
			if (QueryServiceStatus(service, &serviceStatus)) {
				if ( serviceStatus.dwCurrentState == SERVICE_STOPPED )
					DeleteService(service);
			}

			CloseServiceHandle(service);
		} else {
			rc = FALSE;
			errcd = GetLastError();
		}

		CloseServiceHandle( serviceControlManager );
	} else {
		rc = FALSE;
		errcd = GetLastError();
	}

	return rc;
}

BOOL ConfigService(TCHAR *serviceName, DWORD startMode, TCHAR *path)
{
	BOOL rc = TRUE;

	SC_HANDLE serviceControlManager = OpenSCManager( 0, 0, SC_MANAGER_CREATE_SERVICE );

	if (serviceControlManager != NULL) {
		SC_HANDLE service = OpenService( serviceControlManager, serviceName, SERVICE_CHANGE_CONFIG);
		if (service != NULL) {
			ChangeServiceConfig(
				service,
				SERVICE_WIN32_OWN_PROCESS,
				startMode, SERVICE_ERROR_IGNORE, path,
				0, 0, 0, 0, 0, serviceName );
		} else {
			rc = FALSE;
			errcd = GetLastError();
		}

		CloseServiceHandle( serviceControlManager );
	} else {
		rc = FALSE;
		errcd = GetLastError();
	}

	return rc;
}

BOOL QueryService(TCHAR *serviceName)
{
	BOOL rc = TRUE;
	SC_HANDLE service;
	SERVICE_STATUS serviceStatus;
	SC_HANDLE serviceControlManager = OpenSCManager( 0, 0, SC_MANAGER_CONNECT );

	if (serviceControlManager != NULL) {
		service = OpenService(serviceControlManager, serviceName, SERVICE_QUERY_STATUS);
		if (service != NULL) {
			if (QueryServiceStatus(service, &serviceStatus)) {
				char *state = "UNKNOWN";
				switch (serviceStatus.dwCurrentState) {
					case SERVICE_CONTINUE_PENDING: state = "CONTINUE_PENDING"; break;
					case SERVICE_PAUSE_PENDING: state = "PAUSE_PENDING"; break;
					case SERVICE_PAUSED: state = "PAUSED"; break;
					case SERVICE_RUNNING: state = "RUNNING"; break;
					case SERVICE_START_PENDING: state = "START_PENDING"; break;
					case SERVICE_STOP_PENDING: state = "STOP_PENDING"; break;
					case SERVICE_STOPPED: state = "STOPPED"; break;
				}
				printf("%s\n", state);
			} else {
				rc = FALSE;
				errcd = GetLastError();
			}

			CloseServiceHandle(service);
		} else {
			rc = FALSE;
			errcd = GetLastError();
		}

		CloseServiceHandle(serviceControlManager);
	} else {
		rc = FALSE;
		errcd = GetLastError();
	}

	return rc;
}

BOOL StartupService(TCHAR *serviceName)
{
	BOOL rc = TRUE;
	SC_HANDLE service;
	SC_HANDLE serviceControlManager = OpenSCManager( 0, 0, SC_MANAGER_CONNECT );

	if (serviceControlManager != NULL) {
		service = OpenService( serviceControlManager, serviceName, SERVICE_START);
		if (service != NULL) {
			StartService(service, 0, NULL);
			CloseServiceHandle( service );
		} else {
			rc = FALSE;
			errcd = GetLastError();
		}

		CloseServiceHandle( serviceControlManager );
	} else {
		rc = FALSE;
		errcd = GetLastError();
	}

	return TRUE;
}

BOOL StopService(TCHAR *serviceName)
{
	BOOL rc = TRUE;
	SC_HANDLE service;
	SERVICE_STATUS serviceStatus;
	SC_HANDLE serviceControlManager = OpenSCManager( 0, 0, SC_MANAGER_CONNECT );

	if (serviceControlManager != NULL) {
		service = OpenService( serviceControlManager, serviceName, SERVICE_STOP);
		if (service != NULL) {
			ControlService( service, SERVICE_CONTROL_STOP, &serviceStatus );
			CloseServiceHandle( service );
		}
		CloseServiceHandle( serviceControlManager );
	} else {
		return FALSE;
		errcd = GetLastError();
	}

	return TRUE;
}

BOOL parseArgs(int argc, TCHAR* argv[], int *action)
{
	int i;
	int expArgs = 3;	/* expected number of arguments is either 3 or 7 */

	if (argc >= expArgs) {
		if (!lstrcmpi( argv[1], TEXT("create")) || !lstrcmpi( argv[1], TEXT("config"))) {
			expArgs = 7;
		}
	}
	if (argc != expArgs) {
		fprintf(stderr, "ERROR: invalid number of arguments (%d arguments found, expected %d)\n", argc, expArgs);
		return FALSE;
	}

	if (!lstrcmpi(argv[1], TEXT("create"))) {		*action = CREATE;
	} else if (!lstrcmpi( argv[1], TEXT("config"))) {	*action = CONFIG;
	} else if (!lstrcmpi( argv[1], TEXT("delete"))) {	*action = REMOVE;
	} else if (!lstrcmpi( argv[1], TEXT("query"))) {	*action = QUERY;
	} else if (!lstrcmpi( argv[1], TEXT("start"))) {	*action = START;
	} else if (!lstrcmpi( argv[1], TEXT("stop"))) {		*action = STOP;
	} else {
		fprintf(stderr, "ERROR: invalid operation : %s\n", argv[1]);
		return FALSE;
	}

	serviceName = argv[2];

	for (i = 3; i < argc; ++i) {
		if (lstrcmpi( argv[i], TEXT("start=")) == 0) {
			++i;	/* we know to have the correct number of arguments */
			if (lstrcmpi( argv[i], TEXT("auto")) == 0)
				startMode = SERVICE_AUTO_START;
			else
				startMode = SERVICE_DEMAND_START;
		} else
		if (lstrcmpi( argv[i], TEXT("binpath=")) == 0) {
			++i;
			path = argv[i];
		} else {
			fprintf(stderr, "ERROR: Unknown option : %s\n", argv[i]);
			return FALSE;
		}
	}

	return TRUE;
}

int _tmain( int argc, TCHAR* argv[] )
{
	int action = NO_ACTION;
	BOOL rc;
	LPVOID *errmsg = NULL;

	if (!parseArgs(argc, argv, &action)) {
		PrintUsage();
		exit(1);
	 }

	errcd = 0;	/* not really necessary, but initialized data is never wrong */
	switch (action) {
		case CREATE:	rc = InstallService(serviceName, startMode, path);	break;
		case REMOVE:	rc = UninstallService(serviceName);			break;
		case START:	rc = StartupService(serviceName);			break;
		case STOP:	rc = StopService(serviceName);				break;
		case QUERY:	rc = QueryService(serviceName);				break;
		case CONFIG:	rc = ConfigService(serviceName, startMode, path);	break;
		case HELP:
		default:	rc = PrintUsage();					break;
	}
	if (!rc) {
		/* print error message */
		if (errcd != 0) {
			FormatMessage(FORMAT_MESSAGE_FROM_SYSTEM | FORMAT_MESSAGE_ALLOCATE_BUFFER, 
					0, errcd, 0, (LPTSTR) &errmsg, MAX_PATH, 0);
			fprintf(stderr, "Error (%d) : %s\n", errcd, errmsg);
		} else {
			fprintf(stderr, "Unknown Error occurred\n");
		}
		return 1;
	}

	return 0;
}
