/*
** $Id: watchdog.c,v 2.2 2010/08/23 12:11:12 ronald Exp $
**
** Copyright (C) 2002 topIT Informationstechnologie GmbH
**
** $Log: watchdog.c,v $
** Revision 2.2  2010/08/23 12:11:12  ronald
** Removed some (non-critical compiler warnings)
**
** Revision 2.1  2005/04/07 22:01:05  bernhard
** Integrated "NetBSD beta" (WITHOUT WARRANTY!)
**
** Revision 2.0  2004/04/19 15:25:42  ronald
** Version 2.0 Started
**
** Revision 1.2  2003/01/19 20:51:51  ronald
** Usability improvements
**
** Revision 1.1  2002/12/04 21:50:39  ronald
** Watchdog zum Ueberwachen von Prozessen, und scrolllog zum Beherrschen von
** Logfiles zugefuegt.
**
*/

#include <stdlib.h>
#include <stdio.h>
#include <fcntl.h>
#include <signal.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <errno.h>
#include <string.h>

#include "watchdog.h"
#include "common.h"

#define RETRIES 10

static int max_retries = RETRIES;
static char *progname;
static char **parms;
static char *outputfile = (char*) "/dev/null";


void exit_handler(int p)
{
	/* kill child with same signal and terminate */
	if(childpid)
		kill(childpid, p);
	exit(0);
}

void hup_handler(int p)
{
	/* kill child and restart */
	if(childpid) {
		kill(childpid, SIGTERM);
		sleep(3);
		if(childpid)
			kill(childpid, SIGKILL);
	}
}

void chld_handler(int p)
{
	/* get status is done by waitpid() in mainline */
	childpid = 0;
}


int do_the_job(int argc, char *argv[])
{
	pid_t pid;
	int   status;
	int nr_retries = max_retries;

	if((childpid = fork()) > 0) {	/* parent */
		while((pid = waitpid(childpid, &status, 0)) == -1) {
			if(errno == ECHILD) return 0; /* no child */
		}
		return status;
	}
	if(childpid < 0) {		/* can't start a process, we'll sleep
					   for a little while, and try again
					*/
		childpid = 0;		/* just in case we catch a signal */
		sleep(3);
		return -1;
	}

	pid = getppid();
	/* child, now exec the program to be started */
	while(1) {
		execvp(progname, parms);
 		switch(errno) {
			case EPERM:		/* term */
			case E2BIG:		/* term */
			case ENOEXEC:		/* term */
			case EFAULT:		/* term */
			case ENAMETOOLONG:	/* term */
			case ENOENT:		/* term */
			case ENOTDIR:		/* term */
			case EACCES:		/* term */
			case ELOOP:		/* term */
			case EMFILE:		/* term */
			case EINVAL:		/* term */
			case EISDIR:		/* term */
#ifndef BSD
#ifndef AIX
			case ELIBBAD:		/* term */
#endif
#endif
				kill(pid, SIGTERM);
				exit(1);
				break;
			case EIO:		/* retry n */
				nr_retries--;
				if(!nr_retries) {
					kill(pid, SIGTERM);
					exit(1);
				}
				/* FALLTHROUGH */
			case ENOMEM:		/* retry */
			case ETXTBSY:		/* retry */
			case ENFILE:		/* retry */
				sleep((unsigned) (max_retries - nr_retries + 2));
				break;
			default:
				kill(pid, SIGTERM);
				exit(1);
				break;
		}
	}

}

void usage(char *argv[], char *msg)
{
	if(msg != NULL) 
		fprintf(stdout, "Error : %s\n\n", msg);
	fprintf(stdout, "Usage : \n");
	fprintf(stdout, "%s [OPTIONS] cmd cmdargs\n\n", argv[0]);
	fprintf(stdout, "Options:\n");
	fprintf(stdout, "-r NUMBER\tNumber of retries to start process before to give up (default 10)\n");
	fprintf(stdout, "-o FILENAME\tName of file for logging stdout and stderr (default /dev/null)\n");
	fprintf(stdout, "-h         \tThis message\n\n");
	exit(1);
}

int getopts(int argc, char *argv[])
{
	
	int i, j;
	int got_nor = 0;
	int got_of  = 0;
	char *endptr;

	for(i = 1; i < argc; i++) {
		if(!strcmp(argv[i], "-r")) {
			i++;
			if(i >= argc)	usage(argv, (char*) "Expected another parameter (number of retries)");
			if(got_nor)	usage(argv, (char*) "Number of retries specified twice");
			max_retries = (int) strtol(argv[i], &endptr, 10);
			if(*endptr != '\0')	usage(argv, (char*) "Invalid numeric value for number of retries");
			got_nor = 1;
		} else
		if(!strcmp(argv[i], "-o")) {
			i++;
			if(i >= argc)	usage(argv, (char*) "Expected another parameter (outputfile)");
			if(got_of)	usage(argv, (char*) "Outputfile specified twice");
			if(check_name(argv[i], &outputfile))	usage(argv, (char*) "Invalid outputfilename");
			got_of = 1;
		} else
		if(!strcmp(argv[i], "-h")) {
			usage(argv, NULL);
		} else {
			progname = argv[i];
			break;	/* unknown option -> command */
		}
	}
	if(max_retries < 0)	usage(argv, (char*) "Number of retries should be at least 0");
	parms = (char **) malloc((argc - i + 1) * sizeof(char*));
	if(parms == NULL) {
		fprintf(stderr, "Error allocating memory\n");
		return 1;
	}

	for(j = 0;i<argc;i++, j++) {
		parms[j] = argv[i];
	}
	parms[j] = NULL;

	return 0;
}


int main(int argc, char *argv[])
{
	if(getopts(argc, argv)) return 1;	/* Oops, error in commandline */
	if(daemonize(outputfile)) return 1;	/* Oops, error in Deamonize */
	if(sighandling()) return 1;		/* Oops, error in installing signal handlers */

	/* Now do the job, until we're told to quit */
	while(1) {
		do_the_job(argc, argv);
	}
	return 0;
}
